/**
 * \file mp3stream.cpp
 * @short Implementation of class MP3Stream
 */

#include <cassert>
#include "mp3stream.h"
#include "except.h"

#ifdef DEBUG
#include <iostream>
#endif

MP3Stream::MP3Stream(Bitstream &bs, int type) : m_stream(bs)
{
  m_type = type;
  m_ff = m_f = new MP3Frame(this); //First frame is also a dataframe
  m_cutbytes = 0;
}

MP3Stream::~MP3Stream()
{
  while(m_ff) //Delete allocated frames
    m_ff = m_ff->Destroy();
}

MP3Frame *MP3Stream::First() const
{
  return m_f;
}

void MP3Stream::SetFirst(MP3Frame *frame)
{
  assert(frame);
  m_f = frame;
}

Bitstream &MP3Stream::GetBitstream() const
{
  return m_stream;
}

void MP3Stream::WriteData(Bitstream &bs, int numbytes, int startpos)
{
  int cutstart, size, writesize;
  
  while(numbytes > 0 && m_f)
  {
    if(!m_cutbytes) //New frame: calculate it's data offset
      m_f->SetDataBegin(startpos);
    
    size = m_f->DataSize();
    //Calculate beginning of rest of the frame
    cutstart = m_cutbytes ? (size - m_cutbytes) : 0;
    
    //If the rest of frame is bigger than numbytes, cut its end
    if(size - cutstart > numbytes )
      m_cutbytes = size - cutstart - numbytes;
    else
      m_cutbytes = 0;
    
    //Sanity checks
    writesize = size - cutstart - m_cutbytes;
    assert(writesize <= numbytes);
    assert(writesize >= 0);
    
    if(writesize > 0) //Just optimalization
      m_f->WriteData(bs, m_ff, cutstart, m_cutbytes);
    
    numbytes -= writesize;
    startpos += writesize;
    
    try {
      if(!m_cutbytes) //Frame was written completely, go to next
        m_f = m_f->Next();    
    } 
    catch(eEOF e)
    {
      m_f = 0;
    }
  }
  
  if(numbytes > 0)  
  {  
    #ifdef DEBUG
    std::cout << "append " << numbytes << " zeroes" << std::endl;
    #endif
    
    bs.WriteNullBytes(numbytes);
  }
}

void MP3Stream::SeekTo(MP3Frame *frame)
{
  //Seek to given frame, but locked frames can't be deleted
  while(m_ff != frame && !m_ff->isLocked())
  {  
    if(m_f == m_ff) m_f = frame; //Firstrame must be moved too
    m_ff = m_ff->Destroy();
  }
}

int MP3Stream::WriteMode() const
{
  return m_type;
}
