/**
 * \file mp3frame.h
 * @short Declaration of class MP3Frame
 */

#ifndef _MP3FRAME_H_
#define _MP3FRAME_H_

#include "bitstream.h"
#include "frameheader.h"
#include "dataheader.h"
#include "channelheader.h"
#include "audiodata.h"

class MP3Stream;

/** 
 * @short Core class
 *
 * Implements operations with MP3 Frames, creates a linked
 * list of frames, works in cooperation with MP3Stream.
 *
 * Most complexity is hidden here: MP3 files have so-called
 * "bit reservoir". Frame data can be stored in few previous frames,
 * so when a frame is written you have to work with 3 frames:
 * the one you are writing, another one from which you write data
 * and a third one where the data really are.
 *
 * This class handles that, see function Write().
 *
 * @warning Use this class only through MP3Iterator, unneeded
 *          frames could be dealocated without warning.
 */

class MP3Frame
{
  public:
    
    /** Write mode */
    enum { nowrite,   /**< Don't read audio data*/
           fastwrite, /**< Write frames as they were in input (with the same data) */
           normal     /**< Rearrange data in frames (needed for cutting or joining of files */
         };
    
    /**
     * Constructor.
     *
     * Reads data from MP3Stream, skips corrupted frames, so it always constructs
     * an correct frame. EOF condition is handled in Bitstream via exceptions.
     * @param stream MP3Stream to read data from
     * @param prev Previous frame, needed to calculate position in stream and 
     *             sanity checks, can be NULL in case of first frame of file.
     */
    MP3Frame(MP3Stream *stream, MP3Frame *prev = 0);
    
    /**
     * Destructor
     */
    ~MP3Frame(); 
    
    /**
     * Deletes itself and returns next frame.
     *
     * @return Next frame
     */
    MP3Frame *Destroy();
    
    /**
     * Returns next frame (and reads it if it does not exist).
     *
     * @return Next frame in linked list
     */
    MP3Frame *Next();
    
    /**
     * Write the frame with data.
     *
     * Action depends on write mode of MP3Stream. In #fastwrite mode it simply
     * writes data in this frame. In #normal mode it calls MP3Stream::WriteData()
     * with number of bytes it needs.
     *
     * It locks the frame so it won't be deallocated.
     *
     * @param bs Output Bitstream
     */
    void Write(Bitstream &bs);
    
    /**
     * Calculates offset of frame data.
     *
     * The offset is difference between position of frame in stream
     * and the position of its first byte (which can be up to 
     * mp3::max_databegin lower). When the offset should be higher 
     * than mp3::max_databegin, padding bytes are added.
     *
     * @param startpos Position of first byte of data in stream
     */
    void SetDataBegin(int startpos);
    
    /**
     * Changes bitrate of frame in order to resize it.
     *
     * When data in stream are moved forward (eg. when small frames on beginning are 
     * removed), sometimes they would overflow to next frame, which is forbidden.
     * So the frame has to be resized by changing its bitrate.
     *
     * @param num How many bytes must be added
     * @throw cannot_proceed When even the maximum bitrate won't fit all data
     */
    void Enlarge(int num);
    
    /**
     * Gets data belonging to frame from stream (ie. not data in the stream) 
     * and writes them.
     *
     * Parameters \e cutstart and \e cutend are used to write
     * only a part of frame so it fits to data part of other frame.
     * 
     * @warning It automatically deletes unneeded frames (whose data
     *          were already written), so \e frame can become invalid.
     *
     * @param bs Output bitsteam
     * @param frame First frame of stream
     * @param cutstart How many bytes remove from beginning of frame
     * @param cutend How many bytes bytes remove from end of frame
     */
    void WriteData(Bitstream &bs, MP3Frame *frame, int cutstart, int cutend);
    
    /**
     * Makes this frame first in stream.
     *
     * Needed when frames at the beginning are removed, because Write() on some
     * next frame would cause data from removed frames being written. It don't 
     * delete any frames, because they can keep data of the new first frame.
     */
    void SetFirst();
    
    /**
     * Whether frame is locked (can't be deleted)
     *
     * It's was possible that Write() on some frame dealocated that frame,
     * so i had to implement locking mechanism (only used in Write() in #normal
     * mode). MP3Stream::SeekTo() checks whether frame is locked, but not Destroy().
     *
     * @return Whether frame is locked
     */
    bool isLocked() const;
    
    /**
     * Whether there was some corruption \b before frame
     *
     * Constructor always construct a valid frame, but records
     * if there was some problem (with synchronization, invalid header, etc.).
     *
     * @return Whether corruption was detected
     */
    bool CorruptionFound() const;
    
    /**
     * Get frame's bitrate
     *
     * @return Bitrate in kbps
     */
    int Bitrate() const;
    
    /**
     * Number of bytes of data beloning to frame
     *
     * These are not bytes \b in the frame, but data \b belonging 
     * to frame, which can be in a few preceding frames.
     *
     * @return Number of data bytes
     */
    int DataSize() const;
    
    /**
     * Change volume of frame
     *
     * @param n Volume offset, may be negative
     */
    void ChangeGain(int n);
    
    /**
     * Whether frame is completely silent
     *
     * It detects just digital silence acording to data size.
     * Anything else would require decompression of audio.
     *
     * @todo Try to find a more precise silent detection
     * @return Whether frame is silent
     */
    bool isSilent() const; 
    
    /**
     * Write frame information
     *
     * Writes version, layer, bitrate, etc.
     *
     * @param str Output text stream
     */
    void ShowInfo(std::ostream &str) const; 
  
  private:
     //Initialize frame (may fail because of exception, called in loop)
    bool Init(MP3Frame *prev); 
    void Cleanup(); //Delete everything
    void SetZeroes(); //Set all variables to zero
    void WriteHeaders(Bitstream &bs); //Write frame headers

    FrameHeader *m_fh; //Frame header
    DataHeader *m_dh; //Data header (may be mono or stereo)
    ChannelHeader *m_ch[2][2]; //Header of each channel in each granule
    AudioData *m_data; //Raw audio data
    MP3Stream *m_stream; //Stream to read frame from
    MP3Frame *m_next; //Next frame (linked list)
    unsigned int m_CRC; //CRC code (rarely used)
    bool m_locked; //If the frame can't be deleted
    bool m_corruption; //If there was corruption before frame
    bool m_firstframe; //If the frame is first in a file
    int m_spacesize; //Size of part for data (without headers) in bytes
    int m_datasize; //Size of data belonging to frame
    int m_stuffing; //Number of added zero bytes
    int m_databegin; //Offset of first dfata byte in input stream
    int m_newdatabegin; //Offset in output stream
    int m_totalbytes; //Sum of space bytes in previius frames
    int m_framesize; //Total size of frame
    int m_sizechange; //Change of size after resize (bitrate change)
};

#endif //_MP3FRAME_H_
