/**
 * \file mp3core.h
 * @short Declaration of class MP3Core
 */

#ifndef _MP3CORE_H_
#define _MP3CORE_H_

#include "id3tag.h"

/** 
 * @short Top-level class
 * 
 * Implements argument parsing and file 
 * processing, created in and called from main().
 */
class MP3Core
{
  public:  
    
    /**
     * Constructor, initalizes everything to zero.
     */
    MP3Core();
    
    /**
     * Parses command line arguments and fills internal structures.
     * @param argc	Argument count from main()
     * @param argv[]	Arguments from main()
     * @return Whether arguments were correct
     */
    bool Params(int argc, const char *argv[]);
  
    /**
     * Processes the MP3 stream and extracts information (bitrate, number of frames...).
     * Also displays ID3 tags and detects silence (simply ask whether frame is silent, a block of frames 
     * is considered silent when there are no more than mp3::maxloud non-silent frames in a row).
     */
    void GetInfo();
    
    /**
     * Processes input file(s) and writes the result (joined files, removed frames,
     * changes volume etc.). This operation can theoretically fail, because sometimes 
     * data won't fit in frame and it must be resized, which is not possible at maximum
     * bitrate.
     * @return Whether operation was succesful
     */
    bool WriteResult();
    
    /**
     * Do what should be done (extract tags, display info, write data).
     * @return Whether operation was succesful
     */
    bool Process();
    
    /**
     * Display a help message
     * @param filename Name of binary (argv[0])
     */
    static void ShowHelp(const char *filename);
    
  private:
    const char *m_outfile; //Output file
    const char **m_infiles; //Null-terminated array of input files
    bool m_showinfo; //Verbose mode (display bitrate, ID3 tags, etc.)
    int m_gain; //Change of volume
    bool m_autocutstart; //Whether detect & cut silence from beginning
    int m_autocutend; //Whether detect & cut silence from end
    int m_cutstart; //How many frames to remove from beginning
    int m_cutend; //How many frames to remove from end
    int m_numframes; //Number of frames of all input files
    ID3Tag m_tag; //Operations with tags are delegated to this class
};

#endif //_MP3CORE_H_
