/**
 * \file mp3_format.h
 * @short Various constants and tables
 */

#ifndef _MP3_FORMAT_H_
#define _MP3_FORMAT_H_

/**
 * @short Frame header constants
 *
 * Static class.
 */
struct mp3_frame_header
{
    /** Size, without 11 synchronization bits !!! */
    enum { size = 21 };

    /** Symbolic names for fields in bitarray */
    enum { IDex, ID, layer, protection_bit, bitrate_index, frequency, 
           padding_bit, private_bit, mode, mode_extension, copyright, original, emphasis };
    
    /** MP3 stream modes */
    enum { stereo, joint_stereo, dual_channel, single_channel };
           
    static const int bits[]; /**< Positions of fields in Bitarray */
    static const int bitrate_table[2][4][16]; /**< Bitrate according to index and stream type */
    static const int version_table[2][2]; /**< MPEG version table */
    static const int layer_table[4]; /**< MPEG Layer table */
    static const int frequency_table[2][2][4]; /**< Frequency according to index and stream type */
};

#ifdef FRAMEHEADER
const int mp3_frame_header::bits[] = { 0, 1, 2, 4, 5, 9, 11, 12, 13, 15, 17, 18, 19, 21 };
const int mp3_frame_header::bitrate_table[2][4][16] =
  { { { -1, -1, -1, -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, -1 },
      { -1,  8, 16, 24,  32,  64,  80,  56,  64, 128, 160, 112, 128, 256, 320, -1 },
      { -1, 32, 48, 56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384, -1 },
      { -1, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, -1 } },
    { { -1, -1, -1, -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, -1 },
      { -1, 32, 40, 48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, -1 },
      { -1, 32, 48, 56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384, -1 },
      { -1, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, -1 } } };

const int mp3_frame_header::version_table[2][2] = { { 3, -1 }, { 2, 1 } };
const int mp3_frame_header::layer_table[4] = { -1, 3, 2, 1 };
const int mp3_frame_header::frequency_table[2][2][4] = { { { 11025, 12000,  8000, -1 }, {    -1,    -1,    -1, -1 } }, 
                                                         { { 22050, 24000, 16000, -1 }, { 44100, 48000, 32000, -1 } } };
#endif
                                                         
/** 
 * @short Data header constants (MPEG1 mono)
 *
 * Static class.
 */
struct mp3_data_header_mpeg1_1ch
{ 
  /** Size in bits */
  enum { size = 18 }; 
  
  /** Symbolic names for fields in bitarray */
  enum { main_data_begin, private_bits, scfsi }; 
  
  static const int bits[]; /**< Positions of fields in Bitarray */
};

/**
 * @short Data header constants (MPEG1 stereo)
 *
 * Static class.
 */
struct mp3_data_header_mpeg1_2ch
{ 
  /** Size in bits */
  enum { size = 20 };
  
  /** Symbolic names for fields in bitarray */
  enum { main_data_begin, private_bits, scfsi };  
  
  static const int bits[]; /**< Positions of fields in Bitarray */
};

/** 
 * @short Data header constants (MPEG2 mono)
 *
 * Static class.
 */
struct mp3_data_header_mpeg2_1ch
{ 
  /** Size in bits */
  enum { size = 9 }; 
  
  /** Symbolic names for fields in bitarray */
  enum { main_data_begin, private_bits }; 
  
  static const int bits[]; /**< Positions of fields in Bitarray */
};

/**
 * @short Data header constants (MPEG2 stereo)
 *
 * Static class.
 */
struct mp3_data_header_mpeg2_2ch
{ 
  /** Size in bits */
  enum { size = 10 };
  
  /** Symbolic names for fields in bitarray */
  enum { main_data_begin, private_bits };  
  
  static const int bits[]; /**< Positions of fields in Bitarray */
};

#ifdef DATAHEADER
const int mp3_data_header_mpeg1_1ch::bits[] = { 0, 9, 14, 18 };
const int mp3_data_header_mpeg1_2ch::bits[] = { 0, 9, 12, 20 };
const int mp3_data_header_mpeg2_1ch::bits[] = { 0, 8, 9 };
const int mp3_data_header_mpeg2_2ch::bits[] = { 0, 8, 10 };
#endif                                                         


/** 
 * @short Channel header constants (MPEG1)
 *
 * Static class.
 */
struct mp3_channel_header_mpeg1
{
  /** Size in bits */
  enum { size = 59 };
  
  /** Symbolic names for fields in bitarray */
  enum { part2_3_length, big_values, global_gain, scalefac_compress, window_switching_flag, 
         variable_part, preflag, scalefac_scale, count1table_select };
  
  static const int bits[]; /**< Positions of fields in Bitarray */
};

/** 
 * @short Channel header constants (MPEG2)
 *
 * Static class.
 */
struct mp3_channel_header_mpeg2
{
  /** Size in bits */
  enum { size = 63 };
  
  /** Symbolic names for fields in bitarray */
  enum { part2_3_length, big_values, global_gain, scalefac_compress, window_switching_flag, 
         variable_part, scalefac_scale, count1table_select };
  
  static const int bits[]; /**< Positions of fields in Bitarray */
};

#ifdef CHANNELHEADER
const int mp3_channel_header_mpeg1::bits[] = { 0, 12, 21, 29, 33, 34, 56, 57, 58, 59 };
const int mp3_channel_header_mpeg2::bits[] = { 0, 12, 21, 29, 38, 39, 61, 62, 63 };
#endif


/** 
 * @short ID3 Tag V1 constants
 *
 * Static class.
 */
struct mp3_tag_v1
{
  /** Size in bytes */
  enum { size = 128,
         headersize = 3
       }; 
  
  static const char *header; /**< ID3v1 tag identification ("TAG") */
};

#ifdef TAGV1
const char *mp3_tag_v1::header = "TAG";
#endif


/** 
 * @short Miscellaneous MP3 constants
 *
 * Static class. 
 */
struct mp3
{
  enum { sync = 0x7FF, /**< Synchronization (eleven 1 bits) */
         syncbits = 11, /**< Number of synchronization bits */
         CRCbits = 16, /**< Number of bits in CRC */
         frametime = 26, /**< Duration of frame in miliseconds */
         maxloud = 4 /**< Number non-silent blocks which end a silence */
       };
};

#endif //_MP3_FORMAT_H_
