/**
 * \file frameheader.h
 * @short Declaration of class FrameHeader
 */

#ifndef _FRAMEHEADER_H_
#define _FRAMEHEADER_H_

#include "mp3_format.h"
#include "bitarray.h"

/**
 * @short Frame Header (part of MP3Frame)
 */
class FrameHeader : private mp3_frame_header
{
  public:
    
    /**
     * Constructor
     *
     * Reads data bits from Bitstream
     *
     * @param bs	Bitstream to read data from
     */
    FrameHeader(Bitstream &bs);
    
    /**
     * Returns static size of header
     *
     * @return Size of header
     */
    int Size() const;
    
    /**
     * Writes data bits to Bitstream
     *
     * @param bs Output Bitstream
     */
    void Write(Bitstream &bs);
    
    /**
     * @return Frame bitrate
     */
    int Bitrate() const;
    
    /**
     * Returns frame version
     *
     * @return 1:MPEG1, 2:MPEG2, 3:MPEG2.5
     */
    int Version() const; 
    
    /**
     * Returns frame MPEG layer
     *
     * @return 1, 2 or 3
     */
    int Layer() const; 
    
    /**
     * @return Whether copyright bit is set
     */
    bool isCopyrighted() const;
    
    /**
     * @return Whether copy bit is unset
     */
    bool isOriginal() const;
    
    /**
     * @return Whether frame is protected by CRC
     */
    bool isCRC() const;
    
    /**
     * @return Frame frequency in kHz
     */
    int Frequency() const;
    
    /**
     * @return Frame mode as defined in mp3_frame_header (eg. stereo)
     */
    int Mode() const;
    
    /**
     * @return Whether frame is Intensity Stereo (joint stereo mode only)
     */
    bool isIntensStereo() const;
    
    /**
     * @return Whether frame is MS Stereo (joint stereo mode only)
     */
    bool isMSstereo() const;
    
    /**
     * @return Number of channels (1 or 2)
     */
    int NumChannels() const;
    
    /**
     * @return Number of granules (1 or 2)
     */
    int NumGranules() const;

    /**
     * @return Size of frame in bytes
     */
    int FrameSize() const;
    
    /**
     * @return Whether frame header is correct (undamaged)
     * @todo Layer 1 and Layer 2 frames are considered invalid for now
     */
    bool isValid() const;
    
    /**
     * Enlarges frame by changing bitrate to the next in table
     */
    void Enlarge();
    
  private:
    Bitarray<mp3_frame_header> m_fh; //Data (bit struct)
};

#endif //_FRAMEHEADER_H_
