/**
 * \file dataheader.h
 * @short Declaration of classes DataHeaderStereo, DataHeaderMono and DataHeader
 */

#ifndef _DATAHEADER_H_
#define _DATAHEADER_H_

#include "mp3_format.h"
#include "bitarray.h"

/**
 * @short Data Header (part of MP3Frame)
 *
 * The header is different in mono and stereo file
 * and also in MPEG1 and MPEG2, solved by inheritance 
 */
class DataHeader
{
  public:
    
     /**
     * Constructor
     *
     * Reads data bits from Bitstream
     *
     * @param bs	Bitstream to read data from
     */
    DataHeader(Bitstream &bs) {}
   
   /**
     * Returns static size of header
     *
     * @return Size of header
     */
    virtual int Size() const = 0;
    
    /**
     * Writes data bits to Bitstream
     *
     * @param bs Output Bitstream
     */
    virtual void Write(Bitstream &bs) = 0;
    
    /**
     * Returns data offset
     * 
     * (audio data of frame could stored in few previous frames)
     *
     * @return Offset of first data bytes in stream
     */
    virtual int DataBegin() const = 0;
    
    /**
     * Returns maximum value of data offset
     * 
     * @return Maximum value of data offset
     */
    virtual int MaxDataBegin() const = 0;
    
    /**
     * Sets data offset
     * 
     * @param databegin New data offset
     */
    virtual void SetDataBegin(int databegin) = 0;

    /**
     * Destructor
     *
     * Does nothing, only defined virtual to silence GCC warning
     */
    virtual ~DataHeader() {}
};

template<class T> class DataHeaderTemplate : public DataHeader
{
  public:
    DataHeaderTemplate<T>(Bitstream &bs) : DataHeader(bs), m_dh(bs) {}
    int Size() const { return T::size; }
    void Write(Bitstream &bs) { m_dh.Write(bs); }
    int DataBegin() const { return m_dh^T::main_data_begin; }
    int MaxDataBegin() const { return m_dh.Max(T::main_data_begin); }
    void SetDataBegin(int databegin) { m_dh.Set(T::main_data_begin, databegin); }
  
  private:
    Bitarray<T> m_dh; //Data (bit struct)
};

extern template class DataHeaderTemplate<mp3_data_header_mpeg1_1ch>;
extern template class DataHeaderTemplate<mp3_data_header_mpeg1_2ch>;
extern template class DataHeaderTemplate<mp3_data_header_mpeg2_1ch>;
extern template class DataHeaderTemplate<mp3_data_header_mpeg2_2ch>;

typedef DataHeaderTemplate<mp3_data_header_mpeg1_1ch> DataHeaderMPEG1Mono;
typedef DataHeaderTemplate<mp3_data_header_mpeg1_2ch> DataHeaderMPEG1Stereo;
typedef DataHeaderTemplate<mp3_data_header_mpeg2_1ch> DataHeaderMPEG2Mono;
typedef DataHeaderTemplate<mp3_data_header_mpeg2_2ch> DataHeaderMPEG2Stereo;

#endif //_DATAHEADER_H_
