/**
 * \file channelheader.h
 * @short Declaration of class ChannelHeader
 */

#ifndef _CHANNELHEADER_H_
#define _CHANNELHEADER_H_

#include "mp3_format.h"
#include "bitstream.h"
#include "bitarray.h"

/**
 * @short Channel Header (part of MP3Frame)
 *
 * The header is different in mono MPEG1 and MPEG2, 
 * solved by inheritance and some template magic
 */
class ChannelHeader
{
  public:
    
    /**
     * Constructor
     *
     * Reads data bits from Bitstream
     *
     * @param bs	Bitstream to read data from
     */
    ChannelHeader(Bitstream &bs) {}
    
    /**
     * Returns static size of header
     *
     * @return Size of header
     */
    virtual int Size() const = 0;
    
    /**
     * Writes data bits to Bitstream
     *
     * @param bs Output Bitstream
     */
    virtual void Write(Bitstream &bs) = 0;
    
    /**
     * Returns size of data in granule 
     *
     * @return Size of data
     */
    virtual int DataSize() const = 0;
    
    /**
     * Returns volume (global gain) of granule 
     *
     * @return Volume
     */
    virtual int Gain() const = 0;
    
    /**
     * Sets a new volume (global gain)
     *
     * @param n New volume
     */
    virtual void ChangeGain(int n) = 0;
    
    /**
     * Destructor
     *
     * Does nothing, only defined virtual to silence GCC warning
     */
    virtual ~ChannelHeader() {}
};

template<class T> class ChannelHeaderTemplate : public ChannelHeader
{
  public:
    ChannelHeaderTemplate<T>(Bitstream &bs) : ChannelHeader(bs), m_ch(bs) {}
    int Size() const { return T::size; }
    void Write(Bitstream &bs) { m_ch.Write(bs); }
    int DataSize() const { return m_ch^T::part2_3_length; }
    int Gain() const { return m_ch^T::global_gain; }
    void ChangeGain(int n) { m_ch.Set(T::global_gain, (m_ch^T::global_gain) + n); }
  
  private:
    Bitarray<T> m_ch; //Data (bit struct)
};

extern template class ChannelHeaderTemplate<mp3_channel_header_mpeg1>;
extern template class ChannelHeaderTemplate<mp3_channel_header_mpeg2>;

typedef ChannelHeaderTemplate<mp3_channel_header_mpeg1> ChannelHeaderMPEG1;
typedef ChannelHeaderTemplate<mp3_channel_header_mpeg2> ChannelHeaderMPEG2;

#endif //_CHANNELHEADER_H_
