/**
 * \file bitstream.h
 * @short Declaration of class Bitstream
 */

#ifndef _BITSTREAM_H_
#define _BITSTREAM_H_

#include <fstream>

/**
 * @short I/O stream with bit access
 *
 * EOF condition is reported via eof_final exception 
 * Input stream can work with list of files (reads them
 * one by one, throws eof_partial at EOF condition, when 
 * there's at least one more file.
 *
 * It's uses 32bit integers, but should be portable, it 
 * handles little/big endian issues. I can't test it,
 * i have just x86 hardware. Could anyone send 
 * a Sparc workstation? :o)
 */
class Bitstream
{    
  public:  
    
    /** Direction */
    enum { in, /** Input stream */
           out /** Output stream */
         };
    
    /**
     * Constructor for one file
     *
     * @param filename	File to open
     * @param mode	Input/Output
     * @throw open_failed When file does not exist
     */
    Bitstream(const char *filename, int mode);
    
    /**
     * Constructor for list of files
     *
     * @warning Works only for input stream!
     *
     * @param files	Null terminated list of files
     * @param mode	Input/Output
     * @throw open_failed When file does not exist
     */
    Bitstream(const char **files, int mode);
    
    /**
     * Read bits from stream.
     *
     * Bits are stored in integer array, first bit
     * in high-order bit of first integer in array. 
     *
     * @param data	Array to store bits in
     * @param numbits	Number of bits to read
     */
    void Read(unsigned int *data, int numbits);
    
    /**
     * Writes bits to stream.
     *
     * Bits are given in integer array, first bit
     * in high-order bit of first integer in array. 
     *
     * @param data	Array with bits to write
     * @param numbits	Number of bits to read
     */
    void Write(const unsigned int *data, int numbits);
    
    /**
     * Seek forward by number of bits
     *
     * @param numbits	How many bits to move forward
     */
    void Seek(int numbits);
    
    /**
     * Reads from the stream until pattern is found 
     *
     * Consecutive read will get bits immediately after the pattern
     * @warning The pattern is found only when it begins on byte boundary!
     *
     * @param pattern	Pattern to find, last bit is the low-order one
     * @param numbits	Number of bits in pattern
     * @return Number of skipped bytes
     */
    int Find(unsigned int pattern, int numbits);
    
    /**
     * Direct read of bytes
     *
     * Skips bit-handling and reads bytes directly.
     * Also used by Read().
     *
     * @throw eof_partial	EOF of non-last input file
     * @throw eof_final	EOF of last input file
     * @param bytes	Array to store bytes in
     * @param num	Number of bytes to read
     */
    void ReadBytes(char *bytes, int num);
    
    /**
     * Direct write of bytes
     *
     * Skips bit-handling and writes bytes directly.
     * Also used by Write().
     *
     * @param bytes	Array to get bytes from
     * @param num	Number of bytes to write
     */
    void WriteBytes(const char *bytes, int num);
    
    /**
     * Write zero bytes to output 
     *
     * @param num	Number of null
     */
    void WriteNullBytes(int num);
    
  private:
    void Open(const char *filename); //Open file
    void CheckTag(); //If ID3v1 tag is present, decrease filesize by its length
    
    std::fstream m_stream; //C++ I/O stream
    const char **m_files; //Null-terminated array of filenames
    int m_mode; //Input/Output
    int m_size; //Size of input file
    int m_leftbits; //Number of bits kept in m_char
    unsigned char m_char; //Left-over bits, ### must be unsigned for shifts!
    static const char *m_NULL; //Null pointer
};
    
#endif //_BITSTREAM_H_
