const FLAG_WORD =              1 shl  0; { Klicove slovo nebo identifikator }
      FLAG_LEFTSPACE =         1 shl  1; { Oddelene mezerou zleva }
      FLAG_RIGHTSPACE =        1 shl  2; { Oddelene mezerou zprava }
      FLAG_INDENT =            1 shl  3; { Standardni odsazeni }
      FLAG_UNINDENT =          1 shl  4; { Konec odsazeni za slovem }
      FLAG_UNINDENT_BEFORE =   1 shl  5; { Konec odsazeni pred slovem }
      FLAG_SPECIAL_INDENT =    1 shl  6; { Odsazeni v delce slova }
      FLAG_SPECIAL_UNINDENT =  1 shl  7; { Konec odsazeni v delce slova}
      FLAG_NEWLINE_AFTER =     1 shl  8; { Zalomeni radky pred slovem }
      FLAG_NEWLINE_BEFORE =    1 shl  9; { Zalomeni radky za slovem }
      FLAG_BLOCK =             1 shl 10; { Zacatek bloku }
      FLAG_PROC_BLOCK =        1 shl 11; { Zacatek tela procedury }
      FLAG_STRING =            1 shl 12; { Zacatek/konec retezce }
      FLAG_IDENTIFIER =        1 shl 13; { Identif. funkce, promenne apod. }
      FLAG_IDENTIF_STICK =     1 shl 14; { "Prilepi" se k identifikatoru }
      FLAG_SPCIND_NO_BREAK =   1 shl 15; { Nezalamuj radky, je-li specialni odsazeni (kvuli 'of') }
      FLAG_COMMENT =           1 shl 16; { Zacatek/konec komentare }
      FLAG_COMMENT_START =     1 shl 17; { Zacatek komentare }
      FLAG_ALT_COMMENT =       1 shl 18; { Druhy typ komentare: (* }
      FLAG_LINEBREAK_FOUND =   1 shl 19; { Pred slovem bylo zalomeni radku }
      FLAG_2PARTS =            1 shl 20; { Prikaz ma 2 casti (if - else) }
      FLAG_2ND_PART =          1 shl 21; { Druha cast prikazu }
      FLAG_INDENT_IF_BLOCK =   1 shl 22; { Odsazeni jen nasleduje-li blok }
      FLAG_FILE_START =        1 shl 23; { Zacatek souboru }
      FLAG_NO_NEWLINE =        1 shl 24; { Pred slovem nemuze byt zalomeni }
      FLAG_ALLOW_PROC_INDENT = 1 shl 25; { Povoleni odsazeni tela procedury }
      FLAG_DENY_PROC_INDENT =  1 shl 26; { Zakaz odsazeni tela procedury }
      FLAG_ALLOW_NEWLINE =     1 shl 27; { Konec zakazu zalamovani radku }
      FLAG_DENY_NEWLINE =      1 shl 28; { Zakaz zalamovani radku }

      { Retezce a komentare se zpracovavaji jinak, nehledaji se v nich prikazy }
      FLAGS_STRING = FLAG_STRING or FLAG_COMMENT or FLAG_ALT_COMMENT;

      FORM_BLANK = 0; { Zadny format }

      { Identifikator: mezera zleva i zprava, je to slovo a identifikator }
      FORM_IDENTIF = FLAG_LEFTSPACE or FLAG_RIGHTSPACE or
                     FLAG_WORD or FLAG_IDENTIFIER;

      { Cislo: mezera zleva i zprava }
      FORM_NUMBER = FLAG_LEFTSPACE or FLAG_RIGHTSPACE;

      STRING_MAX = 255; { Maximalni delka retezce (dano normou jazyka) }
      CHAR_EOF = #255; { Oznaceni, ze jsme dosli na konec souboru }
      STACK_MAX = 255; { Velikost zasobniku pro odsazovani }
      KEYWORD_FILE = 'keywords.txt'; { Soubor s formatem klicovych slov}

      STANDARD_INDENT = 2; { Standarni odsazeni: 2 mezery }

      { Prvky zasobniku na odsazovani }
      IND_ONELINE = 0; { Odsazeni nasledujici radky }
      IND_2PARTS = 1; { Odsazeni prikazu se 2 castmi (If - Else) }
      IND_BLOCK = 2; { Zarazka: blok }
      IND_NONE = 3; { Zarazka: jednoradkove odsazeni jeste nebylo pouzito }
      IND_PROC = 4; { Zarazka: odsazeni tela procedury }

{ Strom pro vyhledavani klicovych slov }
type PTREE_TYPE = ^TREE_TYPE;
     TREE_TYPE = record { Uzel stromu }
       form: longint; { Format slova konciciho v uzlu}
       f: array [#0..#255] of PTREE_TYPE; { Prechodova funkce }
       end;

var f_in, f_out: text; { Vstupni a vystupni soubor }
    tree: PTREE_TYPE; { Koren stromu }
    identif, number, blank: PTREE_TYPE; { Zakladni uzly stromu }


{ V C vrati 1 znak se souboru F nebo CHAR_EOF, jsme-li na konci souboru }
procedure read_char(var f: text; var c: char);
  begin
  if eof(f) then
    c := CHAR_EOF
  else
    read(f,c);
  end;


{ Vypise N mezer do souboru F }
procedure spaces(var f: text; n: integer);
  var i: integer;

  begin
  for i := 1 to n do
    write(f, ' ');
  end;


{ V S vrati 1 slovo nactene ze souboru F a v C posledni precteny znak  }
{ Navratova hodnota je format tohoto slova                             }
function find_word(var f_in: text; var c: char; var s: string): longint;
  var node: PTREE_TYPE; { Aktualni pozice ve strome }
      linebreak: boolean; { Zda bylo pred slovem zalomeni radku }

  begin
  s := '';
  linebreak := false;

  { Vynechavame bile znaky - pozor, pouze ASCII! }
  while (ord(c) <= 32) do
    begin
    if (c = #10) or (c = #13) then { Zalomeni radku: CR/LF}
      linebreak := true;
    read_char(f_in, c);
    end;

  { Prochazeni stromu }
  node := tree; { Zaciname v koreni }
  { Dokud je prechodova funkce definovana a nejsme na konci souboru }
  { (vse po zacatku retezce je retezec, takze ihned skoncime)       }
  while (node^.f[upcase(c)] <> nil) and ((node^.form and FLAG_STRING) = 0) and
        (c <> CHAR_EOF) and (length(s) < STRING_MAX) do
    begin
    s := s + c; { Nacteny znak se pridava ke slovu }
    node := node^.f[upcase(c)]; { Podle prechodove funkce     }
    read_char(f_in,c);          { prechazime do dalsiho uzlu  }
    end;

  if linebreak then { Pokud jsme nasli zalomeni radku, nastavime priznak }
    find_word := node^.form or FLAG_LINEBREAK_FOUND
  else
    find_word := node^.form; { Vracime format slova }
  end;


{ Prida klicove slovo NEW_WORD s formatem FORM do stromu }
procedure add_word(var new_word: string; form: longint);
  var node: PTREE_TYPE; { Aktualni pozice ve stromu }
      i: word; { Aktualni pozice v NEW_WORD }
      c: char; { Znak slova }

  begin
  node := tree; { Zacneme v koreni stromu }

  { Pridani slova do stromu }
  for i := 1 to length(new_word) do
    begin
    c := upcase(new_word[i]); { Znak se prevede na velka pismena }
    { Pokud slovo (resp. jeho cast) jeste neni ve strome }
    if (node^.f[c] = nil) or (node^.f[c] = identif) then
      begin
      new(node^.f[c]); { Prechodova funkce bude ukazovat na novy uzel }
      node := node^.f[c]; { Prejdeme na tento uzel }

      { Klicove slovo muze pokracovat dalsimi pismeny a cisly }
      { (a pak je to identifikator), ale operator ne          }
      if (form and FLAG_WORD) = 0 then
        node^ := blank^
      else
        node^ := identif^;
      end
    else
      node := node^.f[c]; { Jinak proste pokracujeme dle prechodove funkce }
    end;

  node^.form := form; { Do posledniho uzlu se zapise format slova }
  end;


{ Precte retezec z F_IN a vypise ho do F_OUT beze zmen, v C vrati posledni }
{ nacteny znak, konec retezce je slovo s nastavenymi priznaky FLAGS        }
{ Vrati format slova, ktere retezec ukoncuje                               }
function parse_string(var f_in, f_out: text; var c: char; flags: longint): longint;
  var node: PTREE_TYPE; { Aktualni pozice ve strome }

  begin
  node := tree; { Zaciname v koreni stromu }
  { Pokud nejsme na konci souboru a nalezene }
  { slovo neni konec daneho typu retezce     }
  while (c <> CHAR_EOF) and ((node^.f[upcase(c)] <> nil) or
         ((node^.form and FLAGS_STRING) <> flags)) do
    begin
    if node^.f[upcase(c)] = nil then { Pokud nemuzeme pokracovat,   }
      node := tree^.f[upcase(c)]     { nekoncime (porad jsme uvnitr }
    else                             { retezce), misto toho         }
      node := node^.f[upcase(c)];    { zacneme znovu z korene       }

    write(f_out, c);   { Znaky retezce se vypisuji hned }
    read_char(f_in,c); { TODO: ukladat je, je to nutne pro formatovani }
    end;

  parse_string := node^.form; { Vracime format konce retezce }
  end;


{ Vlozi do zasobniku odsazeni STACK s vrcholem TOP }
{ hodnotu X, zaroven upravuje odsazeni (INDENT)   }
procedure stack_push(x: integer; var indent, top: integer; var stack: array of byte);
  begin
  if (top < STACK_MAX) then { Kontrola preteceni zasobniku }
    begin
    if (x = IND_ONELINE) or (x = IND_2PARTS) then { Tyto prvky skutecne    }
      inc(indent, STANDARD_INDENT);               { meni odsazeni, ostatni }
    inc(top);                                     { jsou jen zarazky       }
    stack[top] := x; { Vlastni pridani do zasobniku }
    end;
  end;


{ Odebere vrchol TOP ze zasobniku odsazeni STACK, }
{ zaroven upravuje odsazeni (INDENT)              }
procedure stack_pop(var indent, top: integer; stack: array of byte);
  begin
  if (top > 0) then { Kontrola podteceni zasobniku }
    begin
    { Tyto prvky skutecne meni odsazeni, ostatni jsou jen zarazky }
    if (stack[top] = IND_ONELINE) or (stack[top] = IND_2PARTS) then
      dec(indent, STANDARD_INDENT);
    dec(top);
    end;
  end;


{ Hlavni procedura provadejici vlastni formatovani          }
{ Cte ze souboru F_IN, vypisuje naformatovany text do F_OUT }
procedure indent(var f_in, f_out: text);
  var c: char; { Znak nacteny ze vstupu }
      s: string; { Slovo nactene ze vstupu }
      form, prev_form: longint; { Format aktualniho a predesleho slova  }
      indent: integer; { Odsazeni naseldujici radky }
      stack: array [0..STACK_MAX] of byte; { Zasobnik odsazeni }
      top: integer; { Vrchol zasobniku }
      special_indent: word; { Odsazeni v delce klicoveho slova }
      proc_indent: boolean; { Povoleni odsazovani tela procedury }
      allow_newline: boolean; { Povoleni zalamovani radek }
      blank_line: integer; { Je-li > 0, bude vlozen prazdny radek }

  begin
  indent := 0;
  special_indent := 0;
  stack[0] := IND_BLOCK; { Blok na dne zasobniku funguje jako zarazka }
  top := 0;
  prev_form := FLAG_FILE_START;
  proc_indent := true;
  allow_newline := true;
  blank_line := 0;

  read_char(f_in,c);
  while c <> CHAR_EOF do { Cteni vsech slov ze vstupnuho souboru }
    begin                { VAROVANI: na poradi operaci zalezi!   }
    form := find_word(f_in, c, s); { Nacteni slova a jeho formatu }

    { Kvuli klicovym slovum Function, Procedure a Var: pokud nejsou na      }
    { zacatku radku (nebo souboru), chovaji se jako normalni identifikatory }
    if (((form and FLAG_PROC_BLOCK) <> 0) or
        ((form and FLAG_SPECIAL_INDENT) <> 0)) and
       (((form and FLAG_NEWLINE_BEFORE) = 0) or not allow_newline) and
       (((prev_form and FLAG_NEWLINE_AFTER) = 0) or not allow_newline) and
       ((prev_form and FLAG_FILE_START) = 0) then
      form := FORM_IDENTIF;

    { Pokud je zakazano odsazovani tela procedury, }
    { zrusi se odpovidajici formatovaci priznaky   }
    if ((form and FLAG_PROC_BLOCK) <> 0) and not proc_indent then
      form := form and not (FLAG_PROC_BLOCK or FLAG_INDENT);

    { Povoleni zalamovani radek }
    if (form and FLAG_ALLOW_NEWLINE) <> 0 then
      allow_newline := true;

    { Zakaz zalamovani radek }
    if (form and FLAG_DENY_NEWLINE) <> 0 then
      allow_newline := false;

    { Mezera se vypise, pokud predchozi slovo ma mit mezeru za }
    { sebou A ZAROVEN aktualni slovo ma mit mezeru pred sebou  }
    { Vyjimka: nektere znaky (^) se "prilepi" na identifikator }
    if (((prev_form and FLAG_RIGHTSPACE) <> 0) and
        ((form and FLAG_LEFTSPACE) <> 0)) and
       (((prev_form and FLAG_IDENTIFIER) = 0) or
        ((form and FLAG_IDENTIF_STICK) = 0)) and
       (((form and FLAG_IDENTIFIER) = 0) or
        ((prev_form and FLAG_IDENTIF_STICK) = 0)) then
      write(f_out, ' ');

    { Konec odsazeni v delce klicoveho slova (Var, Const, Type) }
    if ((form and FLAG_SPECIAL_UNINDENT) <> 0) and (special_indent > 0) then
      begin
      special_indent := 0; { Zrusi se specialni odsazeni }
      inc(blank_line); { Casti Var, Const, Type se oddeli prazdnym radkem }
      end;

    { Odsazeni, pokud nasleduje blok (kvuli ':' v Case) }
    if ((prev_form and FLAG_INDENT_IF_BLOCK) <> 0) and
       ((form and FLAG_BLOCK) <> 0) and
       (special_indent = 0) then
      stack_push(IND_ONELINE, indent, top, stack);

      { Konec radku: je-li povolen a predchozi slovo ma mit }
    { konec radku za sebou NEBO aktualni slovo pred sebou }
    { Vyjimky: komentare, klicove slovo 'Of' (zalomeni    }
    {          jen mimo deklaracni cast), zacatek souboru }
    if allow_newline and
       (((prev_form and FLAG_NEWLINE_AFTER) <> 0) or
        ((form and FLAG_NEWLINE_BEFORE) <> 0)) and
       ((form and FLAG_NO_NEWLINE) = 0) and
       (((prev_form and FLAG_SPCIND_NO_BREAK) = 0) or
        (special_indent = 0)) and
       ((prev_form and FLAG_FILE_START) = 0) and
       ((form and FLAG_COMMENT) = 0) then

      begin
      if (stack[top] = IND_NONE) then { Znacka, ze odsazeni se vztahuje }
        begin                         { jen na nasledujici radek        }
        stack_pop(indent, top, stack);
        { Vyhazujeme prvky ze zasobniku, dokud nenarazime na zarazku }
        { Pokud je slovo 2. cast prikazu (Else), potom skoncime,     }
        { kdyz narazime na odsazeni jeho 1. casti (If)               }
        while (stack[top] <> IND_BLOCK) and (stack[top] <> IND_PROC) and
          (((form and FLAG_2ND_PART) = 0) or (stack[top] <> IND_2PARTS)) do
          stack_pop(indent, top, stack);
        { Odsazeni 1. casti prikazu se zrusi ('Else' je odsazene jako 'If') }
        if (((form and FLAG_2ND_PART) <> 0) and (stack[top] = IND_2PARTS)) then
          stack_pop(indent, top, stack);
        inc(blank_line); { Tam, kde se snizilo odsazeni se vlozi }
        end              { prazdny radek - vypada to dobre :-)   }
      else
        { Pokud nenasleduje slovo, ktere meni odsazeni (While, For) }
        { a nezacal blok, bude odsazen jen nasledujici radek        }
        if ((form and (FLAG_INDENT or FLAG_UNINDENT or FLAG_BLOCK)) = 0) and
           (stack[top] <> IND_BLOCK) and (stack[top] <> IND_PROC) then
          stack_push(IND_NONE, indent, top, stack);

      { Pred radek, ktery zvysuje odsazeni (If, While, For apod.)  }
      { se vlozi prazdny radek, totez u Interface a Implementation }
      if ((form and FLAG_INDENT) <> 0) or
         ((form and FLAG_ALLOW_PROC_INDENT) <> 0) or
         ((form and FLAG_DENY_PROC_INDENT) <> 0) then
        inc(blank_line);

      { Pokud ma byt vlozen prazdny radek a nenasleduje slovo    }
      { snizujici odsazeni (kvuli rade End za sebou) ani 2. cast }
      { prikazu (Else) a neni specialni odsazeni (do deklaraci   }
      { se prazdne radky nevkladaji), vlozime prazdny radek      }
      if (blank_line > 0) and
         ((form and FLAG_UNINDENT) = 0) and
         ((form and FLAG_UNINDENT_BEFORE) = 0) and
         ((form and FLAG_2ND_PART) = 0) and
         (special_indent = 0) then
        writeln(f_out);
      blank_line := 0; { Prazdny radek jsme vlozili, pocitame odznova }

      { Snizeni odsazeni pred slovem (napr. Until): zrusi ze zarazka    }
      { bloku a odsazeni, misto toho se vlozi zarazka odsazeni 1. radku }
      if ((form and FLAG_UNINDENT_BEFORE) <> 0) then
        begin
        stack_pop(indent, top, stack);
        stack_pop(indent, top, stack);
        stack_push(IND_NONE, indent, top, stack);
        inc(blank_line); { Za Repeat - Until je take volny radek }
        end;

      writeln(f_out); { Nyni se konecne vypise zalomeni radku }
      spaces(f_out, indent + special_indent); { Vypisi se mezery }
      end;                                    { tvorici odsazeni }

    { Pokud je pred komentarem zalomeni radku, bude komentar na nove radce }
    { POZNAMKA: Jediny pripad, kde se bere ohled na puvodni  }
    {           formatovani - tady to proste jinak nejde     }
    { VAROVANI: Nejde pripojit k predchozimu: komentare }
    {           nemaji vliv na format vlastniho kodu!   }
    if ((form and FLAG_COMMENT) <> 0) and
       ((form and FLAG_LINEBREAK_FOUND) <> 0) then
      begin
      writeln(f_out);
      spaces(f_out, indent + special_indent);
      end;

    { Odsazeni: do zasobniku se prida prvek podle toho, zda je prikaz }
    { normalni, nebo ma 2 casti (Else se musi priradit spravnemu If)  }
    if ((form and FLAG_INDENT) <> 0) then
      begin
      if (form and FLAG_2PARTS) <> 0 then
        stack_push(IND_2PARTS, indent, top, stack)
      else
        stack_push(IND_ONELINE, indent, top, stack);
      dec(blank_line); { Za radkou zvysujici odsazeni neni prazdny }
      end;             { radek (kvuli napr. nekolika While v sobe) }

    { Pokud slovo zacina blok, vlozi se do zasobniku zarazka }
    if ((form and FLAG_BLOCK) <> 0) then
      begin
      stack_push(IND_BLOCK, indent, top, stack);
      dec(blank_line); { Za zacatkem bloku (Begin) neni prazdny radek }
      end;

    { Procedura ma specialni zarazku (je odsazene cele telo, ale teprve nekde }
    { v nem je Begin - tj. vlastne to jsou 2 bloky, ale konci jen jednim End) }
    if ((form and FLAG_PROC_BLOCK) <> 0)  then
      stack_push(IND_PROC, indent, top, stack);

    { Konec bloku, snizeni odsazeni: zrusi se zarazka bloku a pripadne  }
    { i zarazka procedury, misto toho se vlozi zarazka odsazeni 1 radku }
    if ((form and FLAG_UNINDENT) <> 0) then
      begin
      stack_pop(indent, top, stack);
      if (stack[top] = IND_PROC) then
        stack_pop(indent, top, stack);
      stack_push(IND_NONE, indent, top, stack);
      inc(blank_line); { Za koncem bloku je prazdny radek }
      end;

    { Specialni odsazeni v delce klicoveho slova (+ mezery za nim) }
    if (form and FLAG_SPECIAL_INDENT) <> 0 then
      special_indent := length(s) + 1;

    { "Interface": zakaz odsazovani tel procedur, volny radek  }
    if (form and FLAG_DENY_PROC_INDENT) <> 0 then
      begin
      proc_indent := false;
      inc(blank_line);
      end;

    { "Implementation": povoleni odsazovani tel procedur, volny radek  }
    if (form and FLAG_ALLOW_PROC_INDENT) <> 0 then
      begin
      proc_indent := true;
      inc(blank_line);
      end;

    write(f_out, s); { Konecne se vypise nactene slovo }

    { Pokud slovem zacina retezec nebo komentar, zpracuje se samostatne }
    if ((form and FLAG_STRING) <> 0) or
       ((form and FLAG_COMMENT_START) <> 0) then
      form := parse_string(f_in, f_out, c, form and FLAGS_STRING);

    prev_form := form; { Ulozi se format slova, bude potreba }
    end;               { pro formatovani nasledujiciho slova }
  end;


{ Vytvoreni zakladnich uzlu stromu: koren (TREE), prazdny uzel   }
{ (BLANK) a cyklus pro identifikatory (IDENTIF) a cisla (NUMBER) }
procedure create_tree(var tree, identif, number, blank: PTREE_TYPE);
  var c: char; { Index cyklu }

  begin
  new(tree); { Koren stromu }
  new(identif); { Identifikator (promenna, funkce) }
  new(number); { Cislo (posloupnost cislic) }
  new(blank); { Prazdny uzel (prechodova funkce je nedefonovana) }

  { Nastaveni formatu }
  tree^.form := FORM_BLANK;
  identif^.form := FORM_IDENTIF;
  number^.form := FORM_NUMBER;
  blank^.form := FORM_BLANK;

  { Na zacatku je vsechno identifikator                     }
  { VAROVANI: Ne jen pismena, jinak by se program zacyklil! }
  for c := #0 to #255 do
    tree^.f[c] := identif;
  { Vsechno, co zacina cislici je cislo }
  for c := '0' to '9' do
    tree^.f[c] := number;

  { Identifikatory se skladaji z pismen, cisel a '_' }
  for c := #0 to #255 do
    identif^.f[c] := nil;
  for c := 'A' to 'Z' do      { Bereme jen velka pismena, mala  }
    identif^.f[c] := identif; { pismena se konvertuji na velka  }
  for c := '0' to '9' do
    identif^.f[c] := identif;
  identif^.f['_'] := identif;

  { Cisla se skladaji pouze z cislic }
  { TODO: muze byt problem s float   }
  for c := #0 to #255 do
    number^.f[c] := nil;
  for c := '0' to '9' do
    number^.f[c] := number;

  { Prazdny uzel ma prechodovou funkci nedefinovanou }
  for c := #0 to #255 do
    blank^.f[c] := nil;
  end;


{ Nacita znaky za souboru F, dokud nenarazi na prvni bily         }
{ znak a sklada z nich slovo, ktere vrati v retezci S             }
{ (Pascalovske funkce radsi zkouset nebudu, nejak jim neverim...) }
procedure read_word(var f: text; var s: string);
  var c: char; { Nacteny znak }

  begin
  s := '';

  read(f,c);
  while not eof and (ord(c) > 32) do { Bile znaky jsou ty s kodem <= 32 }
    begin                            { VAROVANI: pouze ASCII!           }
    s := s + c;
    read(f,c);
    end;
  end;


{ Nacte klicova slova ze souboru se jmenem FILENAME }
procedure load_words(filename: string);
  var f: text; { Vstupni soubor }
      s: string; { Nactene slovo }
      flags: longint; { Format slova }

  begin

  {$I-}

  assign(f, filename);
  reset(f);

  if (IOResult <> 0) then { Kontrola: pokud nelze klicova slova nacist, }
    begin                 { program ihned konci, nemuze formatovat      }
    writeln('Fatal: Can''t read keywords (', filename, ')!');
    halt;
    end;

  {$I+}

  while not eof(f) do
    begin
    read_word(f,s); { Nacti slovo }
    readln(f,flags); { Nacti jeho format }
    add_word(s, flags); { A pridej ho do stromu }
    end;

  close(f);
  end;

begin
if ParamCount < 2 then { Vstupni a vystupni soubor musi }
  begin                { byt zadan na prikazove radce   }
  writeln('Justifies a Pascal source.');
  writeln('Usage: INDENT input_file output_file');
  exit;
  end;

create_tree(tree, identif, number, blank); { Vytvori zakladni uzly stromu }
load_words(KEYWORD_FILE); { Prida do stromu slova ze souboru KEYWORD_FILE }

{$I-}

assign(f_in, ParamStr(1));
reset(f_in);
if (IOResult <> 0) then { Kontrola, ze vstupni soubor lze cist }
  begin
  writeln('Can''t read input file "', ParamStr(1), '"!');
  exit;
  end;

assign(f_out, ParamStr(2));
rewrite(f_out);
if (IOResult <> 0) then  { Kontrola, ze do vystupniho souboru lze zapisovat }
  begin
  writeln('Can''t write to output file "', ParamStr(2), '"!');
  close(f_in);
  exit;
  end;

{$I+}

indent(f_in, f_out); { A nyni se uz konecne provede formatovani souboru }

close(f_in);
close(f_out);

end.