//Implementace vstupu a vystupu pro DOS (Borland C)

//Standardni knihovny
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//Platform-dependent (UNIX) knihovny
#include <signal.h>
#include <unistd.h>
#include <curses.h>
#include <sys/time.h>

//Custom libs
#include "tetris.h"
#include "tetris-net.h"

#define SQ0 "  " //Prazdny ctverecek
#define SQ1 "[]" //Znaky tvorici ctverecek
#define SQ2 "##" //Zvyrazneny ctverecek
#define SQ3 "::" //Alternativni ctverecek

int limit = 100; //Casovy limit, po kterem kostka spadne (v setinach sekundy)
int timeslice = -1; //Maximalni doba zablokovani pri cekani na klavesu
int offset; //O kolik je posunute doprava to, co se vykresluje


//Uklid (ukonceni ncurses, zprava na druhou stranu)
static void finish(int sig)
{
  net_send_quit();
  endwin();
  exit(sig);
}


//myaddnstr neorezava na konci radku, i kdyz se to v manualu pise,
//musel jsem napsat vlastni funkci. Souradnice jsou naopak nez v ncurses!
void myaddstr(int x, int y, char *str)
{
  int mx,my;
  int length;
  
  getmaxyx(stdscr,my,mx);
  length = mx - x;
  
  if(length > 0)
    mvaddnstr(y,x,str,length);
}


//Vypis integeru na zadanou pozici s orezanim na konci radku
void myaddint(int x, int y, int i)
{
  char buffer[12];
  
  snprintf(buffer, 12, "%d", i);
  myaddstr(x, y, buffer);
}


//Vykresli natoceni kostky SHAPE na souradnice C, ctverecek bude symbol STR
void draw_shape(T_SHAPE *shape, T_COORD *c, char *str)
{
  int i;

  for(i = 0; i < 4; i++)
    myaddstr(2 * (shape->sq[i].x + c->x) + 1 + offset, shape->sq[i].y + c->y + 1, str);
  
  refresh();
}


//Vykresli radky od Y1 do Y2 (vcetne!) z pole BOARD
void draw_rows(int y1, int y2, T_BOARD board)
{
  int x,y; //Indexy cyklu - radek, sloupec

  for(y = y1; y <= y2; y++)
    for(x = 1; x < 11; x++)
      myaddstr(2 * x + 1 + offset, y + 1, board[x][y] ? SQ1 : SQ0);
  
  refresh();
}


//Vypis statistickych udaju hry GAME
void show_stats(T_GAME *game)
{
  int i;

  if (game->mode == GAME_MATCH || game->mode == GAME_NET)
    i = game->player ? 35 : 30;
  else
    i = 33;

  myaddint(i + 15, 4, game->stats.all_rows);
  myaddint(i + 15, 5, game->stats.rows_by_number[1]);
  myaddint(i + 15, 6, game->stats.rows_by_number[2]);
  myaddint(i + 15, 7, game->stats.rows_by_number[3]);
  myaddint(i + 15, 8, game->stats.rows_by_number[4]);

  myaddint(48, 10, game->stats.all_bricks);
  myaddint(39, 12, game->stats.single_brick[0]);
  myaddint(49, 12, game->stats.single_brick[2]);
  myaddint(39, 14, game->stats.single_brick[3]);
  myaddint(49, 14, game->stats.single_brick[4]);
  myaddint(39, 16, game->stats.single_brick[5]);
  myaddint(49, 16, game->stats.single_brick[6]);
  myaddint(39, 18, game->stats.single_brick[1]);

  if(game->mode == GAME_COMP)
    myaddint(49, 22, game->stats.qual);
  
  refresh();
}


//Oznameni o prohre
void game_over()
{
  myaddstr(9 + offset, 1, "GAME OVER !"); //Ale hral jsi dobre... budes mi chybet...
  refresh();
  sleep(2); //Aby sis to stihnul vychutnat (2 sekundy)
}


//Kresleni ramecku okolo pole
void draw_frame()
{
  int i; //Index cyklu

  for(i = 0; i < 12; i++)
    myaddstr(2 * i + 1 + offset, 25, SQ1);
  
  for(i = 0; i < 25; i++)
  {
    myaddstr(1 + offset, i + 1, SQ1);
    myaddstr(23 + offset, i + 1, SQ1);
  }
  
  refresh();
}


//Nastavi, ze se bude kreslit do okna WINDOW
void select_window(int window)
{
  offset = window ? 55 : 1; //Je-li okno 1, posune se vsechno o 20 znaku
}


//Pripravi obrazovku pro zadanou hru: ramecek okolo pole, statistika apod.
void prepare_screen(T_GAME *game1, T_GAME *game2)
{
  int i; //Index cyklu

  clear();

  //Statisticke informace budou v prave polovine obrazovky
  myaddstr(30,  1, "      STATISTIKA");
  myaddstr(30,  2, "     Rychlost: "); 
  myaddint(45,  2, limit);
  myaddstr(30, 10, "   Kostky celkem:  ");
  myaddstr(30, 12, "     # :       T :  ");
  myaddstr(30, 14, "     S :       Z :  ");
  myaddstr(30, 16, "     L :       J :  ");
  myaddstr(30, 18, "     I :  ");

  i = (game1->mode == GAME_MATCH || game1->mode == GAME_NET) ? 0 : 3;
  myaddstr(30 + i, 4, "Vymazane rady:  ");
  myaddstr(30 + i, 5, "  jednoduse:  ");
  myaddstr(30 + i, 6, "  2 zaroven:  ");
  myaddstr(30 + i, 7, "  3 zaroven:  ");
  myaddstr(30 + i, 8, "  4 zaroven:  ");

  if(game1->mode == GAME_MATCH || game1->mode == GAME_NET)
    myaddstr(30, 3, "               Ty   On");

  if(game1->mode == GAME_COMP) /* Zobrazeni ovladani (nebo vyhodnosti */
  {                            /* pozice, pokud hraje pocitac sam)    */
    myaddstr(30, 20, " Konec: ESC\n");
    myaddstr(30, 22, " Vyhodnost pozice:\n");
  }
  else
  {
    myaddstr(30, 20, " OVLADANI:");
    myaddstr(30, 21, " sipky \x11 \x1F \x10 - posun");
    myaddstr(30, 22, " pismena Z, X - otaceni");
    myaddstr(30, 23, " mezernik - pauza");
    myaddstr(30, 24, " ESC - konec programu");
  }
  
  select_window(0); //Zvoli se pole prvniho hrace
  draw_frame(); //Vykresli se ramecek okolo pole
  show_stats(game1); //Vypis statistickych udaju
  
  if(game1->mode == GAME_MATCH || game1->mode == GAME_NET)
  {
    select_window(1);  /* Vykresli se ramecek       */
    draw_frame();      /* i pro druheho hrace       */
    show_stats(game2); /* a zobrazi jeho statistika */
  }
}


//Cekani na stisk klavesy nebo konec limitu, vrati stisknutou klavesu
int wait()
{
  int ch = 0; //Stisknuta klavesa
  int timeleft = 0; //Maximalni cas cekani na klavesu
  struct timeval time; //Aktualni cas
  static struct timeval falltime; //Cas posledniho padu kostky
  static int initialized = 0; //Jestli je hodnota falltime platna
  
  if(!initialized) /* Aktualni cas se uklada,               */
  {                /* abychom vedeli kdy ma kostka spadnout */
    gettimeofday(&falltime, NULL);
    initialized = 1;
  }
  
  gettimeofday(&time, NULL); //Vypocet casu zbyvajiciho do padu kostky (v milisekundach)
  timeleft = limit * 10 - (time.tv_sec - falltime.tv_sec) * 1000 - (time.tv_usec - falltime.tv_usec) / 1000;
  
  if(timeleft < 0) /* Pokud uz kostka mela spadnout, */
    timeleft = 0;  /* spadne ihned                   */
  
  if(timeslice > 0 && timeleft > timeslice) /* Trivialni kooperativni multitasking:   */
    timeleft = timeslice;                   /* na stisk klavesy se ceka omezenou dobu */
  
  timeout(timeleft); /* Cekani na stisk klavesy */
  ch = getch();      /* (nebo timeout)          */
  
  if(ch == ERR) //Timeout
  {  
    if(timeleft == timeslice) //Pouze "prepnuti kontextu"
      return 0;
    else //Konec limitu, kostka spadla
    {
      gettimeofday(&falltime, NULL);
      return KEY_DOWN; //Trik: pad kostky je totez jako posun dolu
    }
  }
  
  return ch; //Vratime stisknutou klavesu
}


//Clovek polozi kostku BRICK ve hre GAME, pokud SHOW_HELP, zobrazi napovedu
//Funkce vraci 0, pokud ma program skoncit, jinak 1
int select_human(T_BRICK *brick, T_GAME *game, int show_help)
{
  int command; //Prikaz odpovidajici stisknute klavese
  static int newbrick = 1; //Byla zadana nova kostka, resetuje se pozice
  static T_COORD c; //Souradnice kostky (aktualni a puvodni)
  static T_COORD c_help; //Kam by kostku polozil pocitac
  static T_COORD start_pos = START_POS; //Pocatecni pozice, kterou lze prirazovat
  
  if(newbrick) //Nova kostka
  {
    newbrick = 0;
    c = start_pos; //Vychozi pozice (START_POS lze pouzit jen v deklaraci)
    command = CMD_OTHER; //Kostka se nijak neposune
    
    if(show_help)                     /* Pokud se ma zobrazovat napoveda, */
      find_pos(brick, &c_help, game); /* vypocita se pozice pro kostku    */
  }
  else
  {
    switch(wait()) //Misto stisknute klavesy vrati odpovidajici konstantu
    {
      case ' ':      timeout(-1); getch(); return 2; //Pauza
      case KEY_LEFT: command = CMD_LEFT;  break; //Doleva
      case KEY_RIGHT:command = CMD_RIGHT; break; //Doprava
      case KEY_DOWN: command = CMD_DOWN;  break; //Dolu
      case 'z':      command = CMD_RROT;  break; /* Otoceni po smeru   */
      case 'Z':      command = CMD_RROT;  break; /* hodinovych rucicek */
      case 'x':      command = CMD_LROT;  break; /* Otoceni proti smeru */
      case 'X':      command = CMD_LROT;  break; /* hodinovych rucicek  */
      case 13:       command = CMD_FALL;  break; //Dopad kostky
      case 27:       return 0; //Konec
      default:       return 2; //Jina klavesa (nedela nic)
    }    
    
    draw_shape(&brick->rot[c.r], &c, SQ0); //Maze se puvodni pozice kostky
  }
  
  if(!move_brick(command, &c, brick, game->board)) //Kostka dopadla
  {
    if(show_help) //Napoveda se maze
      draw_shape(&brick->rot[c_help.r], &c_help, SQ0);

    draw_shape(&brick->rot[c.r], &c, SQ1); //Kostka se znovu vykresli
    
    if(!set_brick(&brick->rot[c.r], &c, game)) /* Kostka se polozi,     */
    {                                          /* zustala-li ve vychozi */
      game_over();                             /* pozici, hra konci     */
      return 0;
    }

    if(game->last_changed != -1) //Pokud se rady zmenily, vykresli se
    {  
      draw_rows(game->first_changed, game->last_changed, game->board);
      net_send_stats(&game->stats); //Posilaji ze pocty vymazanych rad
    }
    
    net_send_board(game); //Posila se cela hraci plocha
    show_stats(game); //Vypis statistickych udaju
    
    newbrick = 1;
    return 1; //Oznameni, ze kostka dopadla (0 znamena konec programu)
  }
  else
  {    
    if(show_help) //Napoveda - kam by kostku polozil pocitac
      draw_shape(&brick->rot[c_help.r], &c_help, SQ3);
    
    draw_shape(&brick->rot[c.r], &c, SQ1); //Zobrazovani kostky
    net_send_brick(brick->num, &c); //Posila se cislo a pozice kostky
    return 2;
  }
}


//Pocitac polozi kostku BRICK ve hre GAME, pokud SHOW_HELP, zobrazi napovedu
//Funkce vraci 0, pokud ma program zkoncit, jinak 1
int select_comp(T_BRICK *brick, T_GAME *game, int highlight)
{
  T_COORD c = START_POS; //Souradnice kostky a jeji otoceni

  //Vyber pozice pro kostku, vyhodnost se uklada do statistiky
  game->stats.qual = find_pos(brick, &c, game) - c.y;

  if(highlight)
  {  
    draw_shape(&brick->rot[c.r], &c, SQ2); //Napred se polozena kostka zvyrazni
    
    if(wait() == 27) //Ukonceni (nebo i zkraceni cekani)
      return 0;
  }
  
  draw_shape(&brick->rot[c.r], &c, SQ1); //A potom zobrazi normalne

  if(!set_brick(&brick->rot[c.r], &c, game)) /* Kostka se polozi, pokud */
  {                                          /* zustala ve vychozi      */
    game_over();                             /* pozici, hra konci       */
    return 0;
  }

  if(game->last_changed != -1) //Pokud se rady zmenily, vykresli se
    draw_rows(game->first_changed, game->last_changed, game->board);
  
  show_stats(game); //Vypis statistickych udaju
  return 1;
}


//Prijima a zobrazuje stav hry druheho hrace
void select_remote(T_GAME *game, T_BRICK *bricks)
{
  int flags; //Co se zmenilo
  int bricknum; //Cislo aktualni kostky
  T_COORD c; //Souradnice kostky
  static T_BRICK *brick = NULL; //Ukazatel na aktualni kostku
  static T_COORD _c; //Predchozi pozice kostky
  
  //Prijmou se data ze site, vlati bitovou masku zmen
  flags = net_receive(game, &bricknum, &c);
  
  if(flags & FLAG_END) //Druhy hrac ukoncil hru
    finish(0);
    
  if(flags & FLAG_BOARD) /* Zmenila se hraci plocha (dopadla kostka), */
  {                      /* zmeny se uz zapsaly, ale musi se zobrazit */
    draw_rows(0, 24, game->board);
    brick = NULL;
  }
  
  if(flags & FLAG_BRICK) //Souradnice kostky - nutno kontrolovat meze, hlavne u bricknum!
    if(bricknum >= 0 && bricknum < 7 && c.x >= 0 && c.x < 12 &&
       c.y >= 0 && c.y < 25 && c.r >= 0 && c.r < bricks[bricknum].n_rot)
    { //Stara pozice se maze (je-li definovana), nova se vykresli
      if(brick) draw_shape(&brick->rot[_c.r], &_c, SQ0);
      brick = bricks + bricknum;
      draw_shape(&brick->rot[c.r], &c, SQ1);
      _c = c;
    }
  
  if(flags & FLAG_STATS) //Zmena statistik
    show_stats(game);
}


//Inicializace ncurses (zkopirovano z manualu)
void ncurses_init()
{
  initscr();             /* initialize the curses library */
  keypad(stdscr, TRUE);  /* enable keyboard mapping */
  nonl();                /* tell curses not to do NL->CR/NL on output */
  curs_set(0);           /* hide cursor */
  noecho();              /* don't echo input */
  cbreak();              /* take input chars one at a time, no wait for \n */
}


//strcpy ktere alokuje potrebne pole a vraci pointer
char *safestrcpy(char *str)
{
  return strcpy((char*)malloc(strlen(str)), str);
}


//Nacteni paramatru + navazani spojeni
int read_options(int argc, char **argv)
{
  int opt; //Parametr (pismeno)
  int gamemode = GAME_HUMAN; //Zvoleny druh hry
  char *portstr = 0; //Cislo portu jako retezec
  char *remoteaddr = 0; //Adresa druhe strany (muze byt hostname)
  struct timeval tv; //Pro inicializaci nahodneho generatoru
  const char *errmsg; //Chybove hlaseni
  
  /* Parametry: p - cislo portu
   *            c - hra pres sit, pripojeni jako klient
   *            d - demo: pocitac hraje sam
   *            s - hra pres sit, pripojeni jako server
   *            m - hra proti pocitaci
   *            h - hra s "napovedou" 
   *            l - cas mezi pady kostky (v setinach sekundy), default 100
   */
  
  while( (opt = getopt(argc, argv, "p:c:dsmhl:")) != -1 )
    switch(opt) 
    {
      case 'p': portstr = safestrcpy(optarg); break;
      case 'c': gamemode = GAME_NET; remoteaddr = safestrcpy(optarg); break;
      case 'd': gamemode = GAME_COMP; break;
      case 's': gamemode = GAME_NET; break;
      case 'm': gamemode = GAME_MATCH; break;
      case 'h': gamemode = GAME_HELP; break;
      case 'l': limit = atoi(optarg); break;
    }

  gettimeofday(&tv,0); /* Inicializace         */
  mysrand(tv.tv_usec); /* nahodneho generatoru */
  
  if(gamemode == GAME_NET) //Inicializace sitoveho spojeni
  { 
    errmsg = remoteaddr ? net_init_client(remoteaddr,portstr)
                        : net_init_server(portstr);
    if(errmsg)
    {
      printf("Connection failed: %s\n", errmsg);
      exit(1);
    }  
  
    /* Klient posila seed nahodneho generatoru, */
    /* server ho prijme a nastavi               */
    if(remoteaddr)
      net_wait_client(tv.tv_usec);
    else
      mysrand(net_wait_server());
  }
  
  free(portstr);
  free(remoteaddr);
  return gamemode;
}

// -= MAIN =-
int main(int argc, char **argv)
{
  T_BRICK bricks[7]; //Data o kostkach
  T_BRICK *brick = 0; //Vybrana kostka
  T_GAME game1, game2; //Udaje o hre
  int status = 1; //Stav hry (0: konec, 1: dopadla kostka, ostatni: nic se nestalo)
  int gamemode; //Druh hry
  
  if(!read_bricks(bricks)) //Nacteni a vypocet udaju o kostkach
  {
    puts("Nelze precist udaje o kostkach (tetris.dat)!");
    return 1;
  }
    
  gamemode = read_options(argc, argv); //Nacteni parametru, navazani spojeni

  signal(SIGINT, finish); //Korektni ukonceni ncurses po SIGINT
  ncurses_init(); //Prepnuti do ncurses - od ted nelze pouzivat printf apod.!
  
  initialize(&game1);
  initialize(&game2);

  game1.mode = game2.mode = gamemode;
  game1.player = 0;
  game2.player = 1;
  
  prepare_screen(&game1,&game2); //Texty, ramecky atd.
   
  switch(gamemode)
  {
    case GAME_HUMAN: //Standardni hra
      while(status) //Dokud clovek polozil kostku
      {
        if(status == 1)
          brick = new_brick(bricks, &game1);  //Nahodny vyber kostky
        status = select_human(brick, &game1, 0); //Hrac poklada kostku
      }
      break;

    case GAME_COMP: //Pocitac hraje sam
      do {
        brick = new_brick(bricks, &game1); //Nahodny vyber kostky
      } while(select_comp(brick, &game1, 1)); //Dokud pocitac polozil kostku
      break;

    case GAME_HELP: //Pocitac ukazuje, kam by kostku polozil
      while(status) //Dokud clovek polozil kostku
      {
        if(status == 1)
          brick = new_brick(bricks, &game1);  //Nahodny vyber kostky
        status = select_human(brick, &game1, 1); //Hrac poklada kostku
      }
      break; //Dokud clovek polozil kostku, 1 znamena zobrazeni napovedy

    case GAME_MATCH: //Hra cloveka proti pocitaci
      while(1)
      {
        brick = new_brick(bricks, &game1); //Nahodny vyber kostky

        //Clovek polozi kostku
        select_window(0);
        do {
          status = select_human(brick, &game1, 0);
        } while(status == 2);
        if(!status) break;

        //Pocitac polozi kostku
        select_window(1);
        if(!select_comp(brick, &game2, 0))
          break;
      }
      break;
    
    case GAME_NET:
      timeslice = 10;
      
      while(status) //Dokud clovek polozil kostku
      {
        if(status == 1)
          brick = new_brick(bricks, &game1);  //Nahodny vyber kostky
        
        select_window(1);
        select_remote(&game2,bricks); //Aktualizace okna druheho hrace
        
        select_window(0);
        status = select_human(brick, &game1, 0); //Bezi max. TIMESLICE milisekund
      }
      break;
  }
  
  finish(0);
  return 0;
}

